using UnityEngine;
using System.Collections;
using System.Globalization;

//This file is for reference only
//The only requirements are that the GameObject name is 'InputManager'
//And that there are the following callbacks defined:
//set_Button( string )
//clear_Button( string )
//controller_Event( string )
//analog_Event( string )

//the definition of what the string value should be parsed as is defined below
//The rest of this file is just an example implementation of how we are using UnityInputUnifier
//Feel free to change anything as long as you maintain the above functions as they are what the 
//unifier sends to Unity from the service.   If you would like to change the way the unifier works,
//then you will need to build the jar yourself with your changes on that end.

public class InputManager : MonoBehaviour
{
	//input axes
	public const int IA_X = 1;
	public const int IA_Y = 2;
	public const int IA_FIREX = 3;
	public const int IA_FIREY = 4;
	public const int IA_X_ANALOG = 5;
	public const int IA_Y_ANALOG = 6;
	
	//input buttons
	public const int IB_LEFT = 1;
	public const int IB_RIGHT = 2;
	public const int IB_FORWARD = 3;
	public const int IB_BACK = 4;
	public const int IB_MOVEMENT = 5;
	public const int IB_FIRE = 6;
	public const int IB_SELECT = 7;
	public const int IB_START = 8;
	public const int IB_L2 = 9;
	public const int IB_R2 = 9;
	
	//button indices
	private const int BUTTON_A = 0;
	private const int BUTTON_B = 1;
	private const int BUTTON_Y = 2;
	private const int BUTTON_X = 3;
	private const int BUTTON_DOWN = 4;
	private const int BUTTON_UP = 5;
	private const int BUTTON_LEFT = 6;
	private const int BUTTON_RIGHT = 7;
	private const int BUTTON_R1 = 8;
	private const int BUTTON_L1 = 9;
	private const int BUTTON_R2 = 10;
	private const int BUTTON_SELECT = 11;
	private const int BUTTON_START = 12;
	private const int BUTTON_L2 = 13;
	private const int BUTTON_ANALOG_LEFT_UP = 14;
	private const int BUTTON_ANALOG_LEFT_LEFT = 15;
	private const int BUTTON_ANALOG_LEFT_DOWN = 16;
	private const int BUTTON_ANALOG_LEFT_RIGHT = 17;
	private const int BUTTON_ANALOG_RIGHT_UP = 18;
	private const int BUTTON_ANALOG_RIGHT_LEFT = 19;
	private const int BUTTON_ANALOG_RIGHT_DOWN = 20;
	private const int BUTTON_ANALOG_RIGHT_RIGHT = 21;
	private const int BUTTON_R3 = 22;
	private const int BUTTON_END = 23;
	
	//max number of players
	private const int MAX_PLAYERS = 2;
	
	//controller states
	private const int BUTTON_HELD = 0;
	private const int BUTTON_PRESSED = 1;
	private const int BUTTON_STATES = 2;
	private bool[, ,] controllerState = new bool[BUTTON_STATES, MAX_PLAYERS, BUTTON_END];
	
	//analog indices
	private const int ANALOG_LX = 0;
	private const int ANALOG_LY = 1;
	private const int ANALOG_TRIGGER_START = 2;	//beyond this values have only one axis
	private const int ANALOG_LT = 2;
	private const int ANALOG_RT = 3;
	private const int ANALOG_END = 4;
	
	//analog data
	private float[,] analogState = new float[2,ANALOG_END];
	private float[,] analogValue = new float[2,ANALOG_END];
	private bool[,] analogComputed = new bool[2,ANALOG_END];
	
	//analog computations
	private const float analogMaxRadius = 120f;
	private const float analogMaxRadiusSq = analogMaxRadius * analogMaxRadius;
	private const float analogMaxRadiusRecip = 1f / analogMaxRadius;
	private const float analogCardinalWidth = 25f;
	private const float analogDiagonalWidth = 15.9f;	// = sqrt((.9*analogCardinalWidth)^2 / 2), length of triangle side where len(hyp) = .9*analogCardinalWidth (full width seemed too wide)
	private const float analogTriggerMax = 255f;
	private const float analogTriggerMaxRecip = 1f / analogTriggerMax;
	
	// Use this for initialization
	void Awake()
	{
		if (instance != null)
		{
			Debug.LogError("Destroying InputManager that was created after singleton, use InputManager.Instance");
			Destroy(gameObject);
		}
	}

	void Start()
	{
#if UNITY_IPHONE || UNITY_ANDROID
		Screen.sleepTimeout = SleepTimeout.NeverSleep;
#endif

		for (int i = 0; i < BUTTON_STATES; ++i)
		{
			for( int j = 0 ; j < MAX_PLAYERS ; ++j )
			{
				for( int k = 0 ; k < BUTTON_END ; ++k )
				{
					controllerState[i,j,k] = false;
				}
			}
			
			for( int j = 0 ; j < ANALOG_END ; ++j )
			{
				analogState[i,j] = 0f;
				analogValue[i,j] = 0f;
				analogComputed[i,j] = false;
			}
		}
	}
	
	public void Update()
	{
	}
	
	public void LateUpdate()
	{
		for( int j = 0 ; j < MAX_PLAYERS ; ++j )
		{
			for( int k = 0 ; k < BUTTON_END ; ++k )
			{
				controllerState[BUTTON_PRESSED,j,k] = false;
			}
		}
	}
	
	private void ButtonResponse( int index, int player, bool down )
	{
		controllerState[BUTTON_HELD,player,index] = down;
		if( down )
		{
			controllerState[BUTTON_PRESSED,player,index] = down;
		}
	}
	
	private void AnalogResponse( int index, int player, float x, float y )
	{
		if( index < ANALOG_TRIGGER_START )
		{
			if( x != analogState[player, index] || y != analogState[player, index+1] )
			{
				analogState[ player, index ] = x;
				analogComputed[ player, index ] = false;
				analogState[ player, index + 1 ] = y;
				analogComputed[ player, index + 1 ] = false;
			}
		}
		else
		{
			if( x != analogState[player, index] )
			{
				analogState[ player, index ] = x;
				analogComputed[ player, index ] = false;
			}
		}
	}
	
	private void SetButton(string encodedVal, int player, bool down)
	{
		switch( encodedVal )
		{
			case "A":	ButtonResponse(BUTTON_A, player, down);				break;
			case "B":	ButtonResponse(BUTTON_B, player, down);				break;
			case "Y":	ButtonResponse(BUTTON_Y, player, down);				break;
			case "X":	ButtonResponse(BUTTON_X, player, down);				break;
			case "D":	ButtonResponse(BUTTON_DOWN, player, down);			break;
			case "U":	ButtonResponse(BUTTON_UP, player, down);			break;
			case "L":	ButtonResponse(BUTTON_LEFT, player, down);			break;
			case "R":	ButtonResponse(BUTTON_RIGHT, player, down);			break;
			case "L1":	ButtonResponse(BUTTON_L1, player, down);			break;
			case "L2":	ButtonResponse(BUTTON_L2, player, down);			break;
			case "R1":	ButtonResponse(BUTTON_R1, player, down);			break;
			case "R2": ButtonResponse(BUTTON_R2, player, down);				break;
			case "R3": ButtonResponse(BUTTON_R3, player, down);				break;
			case "SEL": ButtonResponse(BUTTON_SELECT, player, down);		break;
			case "ST":	ButtonResponse(BUTTON_START, player, down);			break;
			case "LAD": ButtonResponse(BUTTON_ANALOG_LEFT_DOWN, player, down); break;
			case "LAU": ButtonResponse(BUTTON_ANALOG_LEFT_UP, player, down); break;
			case "LAL": ButtonResponse(BUTTON_ANALOG_LEFT_LEFT, player, down); break;
			case "LAR": ButtonResponse(BUTTON_ANALOG_LEFT_RIGHT, player, down); break;
			case "RAD": ButtonResponse(BUTTON_ANALOG_RIGHT_DOWN, player, down); break;
			case "RAU": ButtonResponse(BUTTON_ANALOG_RIGHT_UP, player, down); break;
			case "RAL": ButtonResponse(BUTTON_ANALOG_RIGHT_LEFT, player, down); break;
			case "RAR": ButtonResponse(BUTTON_ANALOG_RIGHT_RIGHT, player, down); break;
		}
	}
	
	public void set_Button(string encodedVal)
	{
		SetButton(encodedVal.Substring(1), int.Parse(encodedVal.Substring (0,1)) - 1, true);
	}

	public void clear_Button(string encodedVal)
	{
		SetButton(encodedVal.Substring(1), int.Parse(encodedVal.Substring (0,1)) - 1, false);
	}
	
	//This function is called to alert the game of controller connections and disconnections
	public void controller_Event( string eventCode )
	{
		//EventCodes are
		//#XC Player # Connected
		//#XD Player # Disconnected
	}
	
	public void analog_Event( string encodedVal )
	{
		//Debug.Log("getting analog event: " + encodedVal);
		string[] vals = encodedVal.Split(',');
		//Debug.Log("getting message: " + vals[0] + " " + vals[1] + " " + vals[2] + " " + vals[3]);

		switch( vals[1] )
		{
			case "LA":	AnalogResponse(ANALOG_LX, int.Parse(vals[0])-1, float.Parse(vals[2]), float.Parse(vals[3]));	break;
			//RA is right analog, but this code doesn't use it explicitly
			//case "RA":	Debug.Log("Got Right Analog info");	break;
			case "L2A":	AnalogResponse(ANALOG_LT, int.Parse(vals[0])-1, float.Parse(vals[2]), 0f);	break;
			case "R2A":	AnalogResponse(ANALOG_RT, int.Parse(vals[0])-1, float.Parse(vals[2]), 0f);	break;
		}
	}
	
	public float GetAxis( int axis, int player )
	{
		float val = 0f;

		switch( axis )
		{
			case(IA_X):
				return (controllerState[BUTTON_HELD, player - 1, BUTTON_LEFT] || controllerState[BUTTON_HELD, player - 1, BUTTON_ANALOG_LEFT_LEFT] ? 1 : 0)
					+ (controllerState[BUTTON_HELD, player - 1, BUTTON_RIGHT] || controllerState[BUTTON_HELD, player - 1, BUTTON_ANALOG_LEFT_RIGHT] ? -1 : 0);
			case IA_X_ANALOG:
				ComputeAnalogStick(ANALOG_LX, player-1);
				val = -analogValue[player-1, ANALOG_LX] + (controllerState[BUTTON_HELD,player-1,BUTTON_LEFT]?1f:0f) + (controllerState[BUTTON_HELD,player-1,BUTTON_RIGHT]?-1f:0f);
				return Mathf.Min( Mathf.Max(val, -1f), 1f);
			case (IA_Y):
				{
					return ((controllerState[BUTTON_HELD, player - 1, BUTTON_DOWN] || 
						     controllerState[BUTTON_HELD, player - 1, BUTTON_ANALOG_LEFT_DOWN]) ? 1 : 0)
						+ ((controllerState[BUTTON_HELD, player - 1, BUTTON_UP] || 
						    controllerState[BUTTON_HELD, player - 1, BUTTON_ANALOG_LEFT_UP]) ? -1 : 0);
				}
			case IA_Y_ANALOG:
				ComputeAnalogStick(ANALOG_LX, player-1);	//always compute from first axis
				val = -analogValue[player-1, ANALOG_LY] + (controllerState[BUTTON_HELD,player-1,BUTTON_DOWN]?1f:0f) + (controllerState[BUTTON_HELD,player-1,BUTTON_UP]?-1f:0f);
				return Mathf.Min( Mathf.Max(val, -1f), 1f);
			case(IA_FIREX):
				return (controllerState[BUTTON_HELD,player-1,BUTTON_ANALOG_RIGHT_LEFT]?1:0) + 
					   (controllerState[BUTTON_HELD,player-1,BUTTON_ANALOG_RIGHT_RIGHT]?-1:0);
			case(IA_FIREY):
				return (controllerState[BUTTON_HELD, player - 1, BUTTON_ANALOG_RIGHT_UP] ? -1 : 0) + 
					   (controllerState[BUTTON_HELD, player - 1, BUTTON_ANALOG_RIGHT_DOWN] ? 1 : 0);
		}
		return 0;
	}
	
	private bool GetKey( int type, int button, int player )
	{
		switch( button )
		{
			case(IB_LEFT):
				return controllerState[type,player-1,BUTTON_LEFT];
			case(IB_RIGHT):
				return controllerState[type,player-1,BUTTON_RIGHT];
			case(IB_FORWARD):
				return controllerState[type, player - 1, BUTTON_UP] || controllerState[type, player - 1, BUTTON_R1] || controllerState[type, player - 1, BUTTON_ANALOG_LEFT_UP];
			case(IB_BACK):
				return controllerState[type, player - 1, BUTTON_DOWN] || controllerState[type, player - 1, BUTTON_L1] || controllerState[type, player - 1, BUTTON_ANALOG_LEFT_DOWN];
			case(IB_MOVEMENT):
				return controllerState[type,player-1,BUTTON_LEFT] || controllerState[type,player-1,BUTTON_RIGHT]
					|| controllerState[type,player-1,BUTTON_UP] || controllerState[type,player-1,BUTTON_DOWN]
					|| controllerState[type, player - 1, BUTTON_ANALOG_LEFT_LEFT] || controllerState[type, player - 1, BUTTON_ANALOG_LEFT_RIGHT]
					|| controllerState[type, player - 1, BUTTON_ANALOG_LEFT_UP] || controllerState[type, player - 1, BUTTON_ANALOG_LEFT_DOWN];
			case(IB_FIRE):
				return controllerState[type, player - 1, BUTTON_L2] || controllerState[type, player - 1, BUTTON_R2] || controllerState[type, player - 1, BUTTON_R3]
					|| controllerState[type, player - 1, BUTTON_A] || controllerState[type, player - 1, BUTTON_B]
					|| controllerState[type, player - 1, BUTTON_Y] || controllerState[type, player - 1, BUTTON_X];
			case (IB_SELECT):
				return controllerState[type,player-1,BUTTON_SELECT];
			case(IB_START):
				return controllerState[type,player-1,BUTTON_START];
			case(IB_R2):
				return controllerState[type,player-1,BUTTON_R2];
		}
		return false;
	}
	
	public bool GetKey( int button, int player )
	{
		return GetKey(BUTTON_HELD, button, player);
	}
	
	public bool GetKey( int button )
	{
		bool val = false;
		for( int i = 1 ; !val && i <= MAX_PLAYERS ; ++i )
		{
			val |= GetKey(button, i);
		}
		return val;
	}
	
	public bool GetKeyPressed( int button, int player )
	{
		return GetKey(BUTTON_PRESSED, button, player);
	}

	public bool GetKeyPressed( int button )
	{
		bool val = false;
		for( int i = 1 ; !val && i <= MAX_PLAYERS ; ++i )
		{
			val |= GetKeyPressed(button, i);
		}
		return val;
	}
	
	public void ClearController( int player )
	{
		--player;
		
		for( int i = 0 ; i < BUTTON_STATES ; ++i )
		{
			for( int k = 0 ; k < BUTTON_END ; ++k )
			{
				controllerState[i,player,k] = false;
			}
		}

		for( int i = 0 ; i < ANALOG_END ; ++i )
		{
			analogState[player, i] = 0f;
			analogValue[player, i] = 0f;
			analogComputed[player, i] = true;
		}
	}

	//analog input
	private void ComputeAnalogStick( int index, int player )
	{
		if( analogComputed[player, index] )
		{
			return;
		}
		
		float x = analogState[player, index];
		float y = analogState[player, index+1];
		float val;
		
		//bool reportOutput = false;
		
		//if applicable, snap axis values to a direction
		if( x > -analogCardinalWidth && x < analogCardinalWidth )
		{
			x = 0f;
		}
		if( y > -analogCardinalWidth && y < analogCardinalWidth )
		{
			y = 0f;
		}
		if( x != 0f && y != 0f )
		{
			//check diagonals
			if( (x - y) > -analogDiagonalWidth && (x - y) < analogDiagonalWidth )
			{
				//Debug.Log("anin: found positive diagonal from x: " + x + "  y: " + y);
				
				//snap to the diagonal axis, take the average of the values
				x = (x + y) * .5f;
				y = x;
			}
			else if( (x + y) > -analogDiagonalWidth && (x + y) < analogDiagonalWidth )
			{
				//Debug.Log("anin: found negative diagonal from x: " + x + "  y: " + y);

				//snap to the negative diagonal axis, average with the negative
				x = (x - y) * .5f;
				y = -x;
			}
			/*
			else
			{
				//Debug.Log("anin: found neither diagonal from x: " + x + "  y: " + y);
				reportOutput = true;
			}
			*/
		}
		
		//normalize the new values from [-1,1]
		val = (x * x) + (y * y);
		if( val > analogMaxRadiusSq )
		{
			val = 1f / Mathf.Sqrt(val);
			x *= val;
			y *= val;
		}
		else
		{
			x *= analogMaxRadiusRecip;
			y *= analogMaxRadiusRecip;
		}
		
		analogValue[player, index] = x;
		analogValue[player, index+1] = y;
		
		analogComputed[player, index] = true;
		analogComputed[player, index+1] = true;
		/*
		if( reportOutput )
		{
			Debug.Log("anin: final output x: " + x + "  y: " + y);
		}
		*/
	}
	
	private void ComputeAnalogTrigger( int index, int player )
	{
		if( analogComputed[player, index] )
		{
			return;
		}
		
		analogValue[player, index] = analogState[index, player] * analogTriggerMaxRecip;
	}
	
	private static InputManager instance = null;
	private static GameObject singletonGO = null;
	public static InputManager Instance
	{
		get
		{
			if (instance == null)
			{
				Debug.Log("Creating InputManager");
				singletonGO = new GameObject();
				instance = singletonGO.AddComponent<InputManager>();
				singletonGO.name = "InputManager";
				DontDestroyOnLoad(singletonGO);
			}

			return instance;
		}
	}
}
